/*jshint esversion: 6 */
/* global define */
define(["lib/Zoot", "src/utils", "lib/tasks", "src/math/mathUtils", "src/math/mat3", "src/math/vec2",
	"lib/polyDecomp/decomp.min", "lib/hull/hull", "lib/tasks/internal", "lodash"],
function(Z, utils, tasks, mathUtils, mat3, vec2,
	polyDecomp, hull, tasksInternal, lodash) {

"use strict";

// Using require here instead of putting box2d in the define call above.
// There were problems with garbage collection when it was in define, possibly due to
// requirejs caching.
let box2d = [];
require(["./lib/box2d/main"], function(b) {box2d = b;});

const kPrefix = "Adobe.Box2d",
	  kNone = 0, // TODO: share these with the Box2dWorld behavior
	  kCreate = 1,
	  kDestroy = 2;

/*
 * Layer Helpers
 */

function getParentBodyLayer (childBodyLayer) {
	return childBodyLayer.parentPuppet.getParentLayer().getWarperLayer();
}

// Handle Helpers

function addToBodyList (layer, params) {
	let tBox2d = layer.getSharedSceneData(kPrefix);

	if (!tBox2d) {
		tBox2d = {
			bodies : new Map()
		};
	}

	tBox2d.bodies.set(params.layer, params);
	layer.setSharedSceneData(kPrefix, tBox2d);
}

function getBodyMap (layer) {
	var tBox2d = layer.getSharedSceneData(kPrefix);

	if (tBox2d) {
		return tBox2d.bodies;
	}

	return undefined;
}

function updateBodyMap (layer, mBodies) {
	let tBox2d = layer.getSharedSceneData(kPrefix);

	tBox2d.bodies = mBodies;

	layer.setSharedSceneData(kPrefix, tBox2d);
}

function cloneBodyContainer (bi) {
	return { body : false,
			 layer : bi.layer,
			 trackItem : bi.trackItem,
			 paramOnly : bi.paramOnly,
			 density : bi.density,
			 dynamic : bi.dynamic,
			 collidable : bi.collidable,
			 autoAddChildren : bi.autoAddChildren,
			 deformable : bi.deformable,
			 motorStrength : bi.motorStrength,
			 motorBody : undefined,
			 source : bi.source,
			 approxShape : bi.approxShape,
			 friction : bi.friction,
			 restitution : bi.restitution,
			 wallCollide : bi.wallCollide,
			 beingAltered : new Map(),
			 action : bi.action,
			 controlJoints : new Map(),
			 controlJointBody : undefined,
			 beingDragged : new Map(),
			 gravVec : bi.gravVec
		   };
}

function getSceneScale (args) {
	const scene = args.scene_;
	return Math.max(scene.getPixelWidth(), scene.getPixelHeight()) / 10;
}

function createPolygonShape(vertices) {
	let shape = new box2d.b2PolygonShape();
	let buffer = box2d._malloc(vertices.length * 8);
	let offset = 0;
	for (let i=0;i<vertices.length;i++) {
		box2d.HEAPF32[(buffer + offset) / 4] = vertices[i][0]; // x
		box2d.HEAPF32[(buffer + offset + 4) / 4] = vertices[i][1]; // y
		offset += 8;
	}
	let ptr_wrapped = box2d.wrapPointer(buffer, box2d.b2Vec2);
	shape.Set(ptr_wrapped, vertices.length);
	box2d._free(buffer);
	return shape;
}

// Query for mouse overlap of any bodies
//let queryOverlapAny = (function () {
//    // Initialize the query callback object
//    let queryCallback = new box2d.JSQueryCallback();
//    queryCallback.ReportFixture = function(fixturePtr) {
//        var fixture = box2d.wrapPointer( fixturePtr, box2d.b2Fixture );
//        //if ( fixture.GetBody().GetType() !== box2d.b2_dynamicBody ) //mouse can only drag dynamic bodies
//            //return true;
//        if ( ! fixture.TestPoint( this.m_point ) )
//            return true;
//        this.m_fixture = fixture;
//        return false;
//    };

//    return function (self, box2dMouse) {
//        // Query the world for overlapping objects
//        queryCallback.m_fixture = null;
//        queryCallback.m_point = box2dMouse;

//        // Make a small box
//        let aabb = new box2d.b2AABB();
//        let d = 0.001;
//        let tempVec = new box2d.b2Vec2(box2dMouse.x - d, box2dMouse.y - d);
//        aabb.set_lowerBound(tempVec);
//        tempVec.set_x(box2dMouse.x + d); tempVec.set_y(box2dMouse.y + d);
//        aabb.set_upperBound(tempVec);

//        self.world.QueryAABB(queryCallback, aabb);

//        box2d.destroy(aabb);
//        box2d.destroy(tempVec);

//        return queryCallback.m_fixture;
//    };
//})();

//function addChildPuppet (sdkLayer, dict, bi, sceneLayer) {
//    let layer = sdkLayer.privateLayer;

//    // skip over layers without display container
//    let displayContainer = layer.getDisplayContainer();
//    if (! displayContainer) return;

//    // TODO: double check this, is there a warper container for layers with only one handle?
//    // skip over Layers without warper container
//    let warperContainer = layer.getWarperContainer();
//    if (! warperContainer) return;

//    // skip if no warping
//    let warperContainerCanWarp = warperContainer.canWarp();
//    if (! warperContainerCanWarp) return;

//    if (!dict.has(sdkLayer)) {
//        // Add this body
//        let body = cloneBodyContainer(bi);
//        body.body = false;
//        body.layer = sdkLayer;
//        body.trackItem = sdkLayer.privateLayer.getTrackItem();
//        body.dynamic = true;
//        body.autoAddChildren = false;
//        body.deformable = false;
//        body.source = false;

//        let tBox2d = sceneLayer.getSharedSceneData(kPrefix);
//        tBox2d.bodies.push(body);
//        sceneLayer.setSharedSceneData(kPrefix, tBox2d);
//        dict.set(sdkLayer, undefined);
//    }
//}

// Find the minimum enclosing circle of a set of points
// Input points must be in ccw order
// Returns object {center: [x, y], radius: r}
// Algorithm from "A simple algorithm for computing the smallest enclosing circle", Sven Skyum 1991
function minEnclosingCircle (inputPoints) {
	let points = inputPoints.slice(0); // duplicate the array

	if (points.length < 3) {
		return {center: points[0] || [0, 0], radius: 0};
	}

	// Construct circle through three distinct points
	function threePtCircle (p1, p2, p3) {
		// Helper function distance between points
		function dist (a, b) {
			return Math.sqrt(Math.pow(a[0] - b[0], 2) + Math.pow(a[1] - b[1], 2));
		}

		// First check for duplicate points
		// TODO: Should this be an epsilon check?
		if (p1[0] === p2[0] && p1[1] === p2[1]) {
			const c = [(p1[0] + p3[0]) / 2.0, (p1[1] + p3[1]) / 2.0];

			return {center: c,
					radius: dist(p1, c)};
		} else if (p1[0] === p3[0] && p1[1] === p3[1]) {
			const c = [(p1[0] + p2[0]) / 2.0, (p1[1] + p2[1]) / 2.0];

			return {center: c,
					radius: dist(p1, c)};
		} else if (p2[0] === p3[0] && p2[1] === p3[1]) {
			const c = [(p1[0] + p2[0]) / 2.0, (p1[1] + p2[1]) / 2.0];

			return {center: c,
					radius: dist(p1, c)};
		}

		// [a b; c d] [xc; yc] = [e; f]
		const a = 2 * (p1[0] - p2[0]),
			  b = 2 * (p1[1] - p2[1]),
			  c = 2 * (p2[0] - p3[0]),
			  d = 2 * (p2[1] - p3[1]),
			  e = Math.pow(p1[0], 2) - Math.pow(p2[0], 2) + Math.pow(p1[1], 2) - Math.pow(p2[1], 2),
			  f = Math.pow(p2[0], 2) - Math.pow(p3[0], 2) + Math.pow(p2[1], 2) - Math.pow(p3[1], 2);

		Z.utils.assert((a !== 0.0 || b !== 0.0) && (c !== 0.0 || d !== 0.0), "Error fitting minimum enclosing circle");

		// TODO: just stuff this into a Mat3 and invert instead?
		let xc = 0.0, yc = 0.0;
		if (a !== 0.0) {
			yc = (f - c*e/a) / (d - c*b/a);
			xc = (e - b*yc) / a;
		} else if (c !== 0.0) {
			yc = (e - a*f/c) / (b - a*d/c);
			xc = (f - d*yc) / c;
		} else if (b !== 0.0) {
			xc = (f - e*d/b) / (c - d*a/b);
			yc = (e - a*xc) / b;
		} else { // d != 0.0
			xc = (e - b*f/d) / (a - b*c/d);
			yc = (f - c*xc) / d;
		}

		return {center: [xc, yc], radius: dist(p1, [xc, yc])};
	}

	// Angle between line segments (p1, p2) and (p3, p2)
	function segAngle (p1, p2, p3) {
		const lxa = p1[0] - p2[0],
			  lya = p1[1] - p2[1],
			  lxb = p3[0] - p2[0],
			  lyb = p3[1] - p2[1],
			  na = Math.sqrt(lxa * lxa + lya * lya),
			  nb = Math.sqrt(lxb * lxb + lyb * lyb);

		return Math.acos( (lxa * lxb + lya * lyb) / na / nb );
	}

	let finish = false,
		maxR = 0.0,
		maxAngle = 0,
		pInd = 0;

	// TODO: currently O(n^2), but if we only recompute the radius and angle for neighboring points when removing a point, it would be O(n log n)
	while (!finish) {
		// Find max
		maxR = 0.0;
		maxAngle = 0;

		for (let ind = 0; ind < points.length; ++ind) {
			// input points in ccw order, algorithm looks at them in cw order
			let circle = threePtCircle(points[(ind+1) % points.length], points[ind], points[(ind-1 + points.length) % points.length]);
			let angle  =      segAngle(points[(ind+1) % points.length], points[ind], points[(ind-1 + points.length) % points.length]);

			if (circle.radius > maxR) {
				maxR = circle.radius;
				maxAngle = angle;
				pInd = ind;
			} else if (circle.radius === maxR) {
				if (angle > maxAngle) {
					maxAngle = angle;
					pInd = ind;
				}
			}
		}

		if (maxAngle <= Math.PI / 2.0 || points.length === 3) {
			finish = true;
		} else {
			// Remove the point
			points.splice(pInd, 1);
		}
	}

	let circle = threePtCircle(points[(pInd+1) % points.length], points[pInd], points[(pInd-1 + points.length) % points.length]);

	return circle;
}

// Compute the minimum enclosing arbitrarily oriented box of a convex polygon
// Returns {rect: [p1, p2, p3, p4], area: a}
function minEnclosingBox (points) {
	let minXind = 0,
		maxXind = 0,
		minYind = 0,
		maxYind = 0;

	const eps = 1e-6;

	// First, find min/max x/y points
	for (let i = 0; i < points.length; ++i)
	{
		if (points[i][0] < points[minXind][0] - eps || (Math.abs(points[i][0] - points[minXind][0]) < eps && points[i][1] > points[minXind][1])) {
			minXind = i;
		}

		if (points[i][0] > points[maxXind][0] + eps || (Math.abs(points[i][0] - points[maxXind][0]) < eps && points[i][1] < points[maxXind][1])) {
			maxXind = i;
		}

		if (points[i][1] < points[minYind][1] - eps || (Math.abs(points[i][1] - points[minYind][1]) < eps && points[i][0] < points[minYind][0])) {
			minYind = i;
		}

		if (points[i][1] > points[maxYind][1] || (Math.abs(points[i][1] - points[maxYind][1]) < eps && points[i][0] > points[maxYind][0])) {
			maxYind = i;
		}
	}

	// Now we define the four calipers
	let c = [{p1: points[minXind], p2: [points[minXind][0], points[minXind][1] + 1.0], ind: minXind},
			 {p1: points[maxYind], p2: [points[maxYind][0] + 1.0, points[maxYind][1]], ind: maxYind},
			 {p1: points[maxXind], p2: [points[maxXind][0], points[maxXind][1] - 1.0], ind: maxXind},
			 {p1: points[minYind], p2: [points[minYind][0] - 1.0, points[minYind][1]], ind: minYind}];

	function dist (p1, p2) {
		return Math.sqrt(Math.pow(p1[0] - p2[0], 2) + Math.pow(p1[1] - p2[1], 2));
	}

	function angle (l1, l2) {
		let dp = (l1[1][0] - l1[0][0]) * (l2[1][0] - l2[0][0]) + (l1[1][1] - l1[0][1]) * (l2[1][1] - l2[0][1]);
		return Math.acos( dp / dist(l1[1], l1[0]) / dist(l2[1], l2[0]) );
	}

	function intersection (l1, l2) {
		// The intersection point of two lines
		const p1 = l1[0],
			  p2 = l1[1],
			  p3 = l2[0],
			  p4 = l2[1],
			  det = (p1[0] - p2[0]) * (p3[1] - p4[1]) - (p1[1] - p2[1]) * (p3[0] - p4[0]);

		Z.utils.assert(Math.abs(det) > 1e-12, "parallel lines");

		let px = ((p1[0] * p2[1] - p1[1] * p2[0]) * (p3[0] - p4[0]) - (p1[0] - p2[0]) * (p3[0] * p4[1] - p3[1] * p4[0])) / det,
			py = ((p1[0] * p2[1] - p1[1] * p2[0]) * (p3[1] - p4[1]) - (p1[1] - p2[1]) * (p3[0] * p4[1] - p3[1] * p4[0])) / det;

		return [px, py];
	}

	function caliperRect(calipers) {
		return [intersection( [calipers[0].p2, calipers[0].p1], [calipers[3].p2, calipers[3].p1] ),
				intersection( [calipers[3].p2, calipers[3].p1], [calipers[2].p2, calipers[2].p1] ),
				intersection( [calipers[2].p2, calipers[2].p1], [calipers[1].p2, calipers[1].p1] ),
				intersection( [calipers[1].p2, calipers[1].p1], [calipers[0].p2, calipers[0].p1] )];
	}

	function caliperArea(calipers) {
		let basis1 = [calipers[1].p2[0] - calipers[1].p1[0], calipers[1].p2[1] - calipers[1].p1[1]],
			basis2 = [basis1[1], -basis1[0]];

		let projPt1 = calipers[0].p1[0] * basis1[0] + calipers[0].p1[1] * basis1[1],
			projPt2 = calipers[2].p1[0] * basis1[0] + calipers[2].p1[1] * basis1[1],
			projPt3 = calipers[1].p1[0] * basis2[0] + calipers[1].p1[1] * basis2[1],
			projPt4 = calipers[3].p1[0] * basis2[0] + calipers[3].p1[1] * basis2[1];

		let d1 = Math.abs(projPt1 - projPt2),
			d2 = Math.abs(projPt3 - projPt4);

		return d1 * d2;
	}

	let minRect = caliperRect(c),
		minArea = caliperArea(c),
		totalAngle = 0.0;

	while (totalAngle < Math.PI / 2.0) { // until calipers rotate 90 degrees
		// Find minimum angle
		let minAngle = Math.PI,
			minInd = 0;

		for (let ind = 0; ind < 4; ++ind) {
			let cAngle = angle( [c[ind].p1, c[ind].p2], [points[c[ind].ind], points[(c[ind].ind - 1 + points.length) % points.length]] );

			if (cAngle < minAngle) {
				minAngle = cAngle;
				minInd = ind;
			}
		}

		totalAngle += minAngle;

		let rot = mat3.identity().rotate(-minAngle);
		let basis1 = vec2.transformLinear(rot, [c[1].p2[0] - c[1].p1[0], c[1].p2[1] - c[1].p1[1]]),
			basis2 = [basis1[1], -basis1[0]];

		vec2.normalize(basis1, basis1);
		vec2.normalize(basis2, basis2);

		minXind = 0; minYind = 0; maxXind = 0; maxYind = 0;
		let minXProj = [0,0], minYProj = [0,0], maxXProj = [0,0], maxYProj = [0,0];

		// Project points onto basis and find extreme points
		for (let i = 0; i < points.length; ++i)
		{
			let projPt = [points[i][0] * basis1[0] + points[i][1] * basis1[1], points[i][0] * basis2[0] + points[i][1] * basis2[1]];

			if (projPt[0] < minXProj[0] - eps || (Math.abs(projPt[0] - minXProj[0]) < eps && projPt[1] > minXProj[1]) || i === 0) {
				minXind = i;
				minXProj = projPt;
			}

			if (projPt[0] > maxXProj[0] + eps || (Math.abs(projPt[0] - maxXProj[0]) < eps && projPt[1] < maxXProj[1]) || i === 0) {
				maxXind = i;
				maxXProj = projPt;
			}

			if (projPt[1] < minYProj[1] - eps || (Math.abs(projPt[1] - minYProj[1]) < eps && projPt[0] < minYProj[0]) || i === 0) {
				minYind = i;
				minYProj = projPt;
			}

			if (projPt[1] > maxYProj[1] || (Math.abs(projPt[1] - maxYProj[1]) < eps && projPt[0] > maxYProj[0]) || i === 0) {
				maxYind = i;
				maxYProj = projPt;
			}
		}

		// Update calipers
		c[0].p1 = points[minXind]; c[0].p2 = [c[0].p1[0] + basis2[0], c[0].p1[1] + basis2[1]]; c[0].ind = minXind;
		c[1].p1 = points[maxYind]; c[1].p2 = [c[1].p1[0] + basis1[0], c[1].p1[1] + basis1[1]]; c[1].ind = maxYind;
		c[2].p1 = points[maxXind]; c[2].p2 = [c[2].p1[0] - basis2[0], c[2].p1[1] - basis2[1]]; c[2].ind = maxXind;
		c[3].p1 = points[minYind]; c[3].p2 = [c[3].p1[0] - basis1[0], c[3].p1[1] - basis1[1]]; c[3].ind = minYind;

		// Check area
		let area = caliperArea(c);

		if (area < minArea) {
			minRect = caliperRect(c);
			minArea = area;
		}
	}

	return {rect: minRect, area: minArea};
}

// Split a polygon into non-overlapping pieces
function splitPolygonNonoverlapping (polyPoints) {
	let intersections = [];

	// TODO: speed this up somehow
	for (let i = 0; i < polyPoints.length; ++i) {
		let found = false;

		for (let j = 0; j < polyPoints.length && !found; ++j) {
			if (j === i || j === (i-1+polyPoints.length) % polyPoints.length || j === (i+1) % polyPoints.length) continue;

			let p1 = polyPoints[i],
				p2 = polyPoints[(i+1) % polyPoints.length],
				p3 = polyPoints[j],
				p4 = polyPoints[(j+1) % polyPoints.length];

			if (polyDecomp.lineSegmentsIntersect(p1, p2, p3, p4)) {
				intersections.push(i + 1);
				found = true;
			}
		}
	}

	let output = [];

	if (intersections.length === 0) {
		output.push(polyPoints);
	} else {
		// First segment
		if (intersections[intersections.length-1] === polyPoints.length) {
			if (intersections[0] > 2) {
				output.push(polyPoints.slice(0, intersections[0]));
			}
		} else {
			let tmpArr = polyPoints.slice(0, intersections[0]).concat(polyPoints.slice(intersections[intersections.length-1]));

			if (tmpArr.length > 2) {
				output.push(tmpArr);
			}
		}

		for (let i = 0; i < intersections.length-1; ++i) {
			let start = intersections[i],
				end = intersections[i+1];

			if (end - start > 2) {
				output.push(polyPoints.slice(start, end));
			}
		}
	}

	return output;
}

function polygonArea (points) {
	if (points.length < 3) return 0;

	// Shoelace formula to compute area
	let start = 0,
		end = points.length,
		area = points[end-1][0] * points[start][1] - points[start][0] * points[end-1][1];

	for (let i = start; i < end-1; ++i) {
		area += points[i][0] * points[i+1][1] - points[i+1][0] * points[i][1];
	}

	area = 0.5 * Math.abs(area);
	return area;
}

function createBox2dBodyHelper (self, args, bi, world, geometry) {
	// Define body...
	const bodyDef = new box2d.b2BodyDef();

	let tmpPos = box2d.b2Vec2.from(geometry.position);
	bodyDef.set_position(tmpPos);
	bodyDef.set_angle(geometry.rotation);
	box2d.destroy(tmpPos);

	// Dummy body for control
	// From what I can tell it is just because the joint requires two bodies
	bodyDef.set_type(box2d.b2_staticBody);
	bi.controlJointBody = world.CreateBody(bodyDef);

	// ... properties
	if (bi.dynamic) {
		bodyDef.set_type(box2d.b2_dynamicBody);
	}

	// Create body
	const body = world.CreateBody(bodyDef);

	box2d.destroy(bodyDef);

	function attachShape (shape) {
		// Define and create fixture
		const fixtureDef = new box2d.b2FixtureDef();
		fixtureDef.set_shape(shape);
		if (bi.dynamic) {
			fixtureDef.set_density(bi.density);
		}
		fixtureDef.set_friction(bi.friction);
		fixtureDef.set_restitution(bi.restitution);
		fixtureDef.get_filter().set_categoryBits(0x0002);

		if (bi.wallCollide) {
			fixtureDef.get_filter().set_maskBits(0x0003);
		} else {
			fixtureDef.get_filter().set_maskBits(0x0002);
		}

		body.CreateFixture(fixtureDef);

		box2d.destroy(fixtureDef);
	}

	function createConvexShape (poly) {
		const shape = createPolygonShape(poly);
		if (bi.collidable) {
			attachShape(shape);
		} else {
			// Just using the shape to get mass properties
			let mass = new box2d.b2MassData();
			shape.ComputeMass(mass, bi.density);
			body.SetMassData(mass);
			box2d.destroy(mass);
		}
		box2d.destroy(shape);
	}

	function createCircleShape (circle) {
		let shape = new box2d.b2CircleShape();
		shape.set_m_radius(circle.radius);
		let p = box2d.b2Vec2.from(circle.center);
		shape.set_m_p(p);
		box2d.destroy(p);
		if (bi.collidable) {
			attachShape(shape);
		} else {
			// Just using the shape to get mass properties
			let mass = new box2d.b2MassData();
			shape.ComputeMass(mass, bi.density);
			body.SetMassData(mass);
			box2d.destroy(mass);
		}
		box2d.destroy(shape);
	}

	if (geometry.useCircle) {
		createCircleShape(geometry.circle);
	} else {
		let convexPolys = [];
		// Box2D polygon shapes (one for each convex part) expect at most 8 points
		// For each polygon, split it into pieces so that 3 <= vertices <= 8
		geometry.polygons.forEach(function (poly) {
			function step(i) {return Math.min(7, (poly.length - 7 - i <= 0 ? poly.length - i : poly.length - 3 - i));}
			for (let i = 1; i < poly.length; i += step(i)) {
				let tmpPoly = [poly[0]].concat(poly.slice(i, i + step(i)));
				// Check area again, this process can create degenerate geometry
				if (polygonArea(tmpPoly) > 1e-3) {
					convexPolys.push(tmpPoly);
				}
			}
		});

		convexPolys.forEach(createConvexShape);
	}

	bi.body = body;

	// Attach motor if requested
	if (bi.dynamic) {
		// Create new body which controls the motor (position constraints)
		const mBodyDef = new box2d.b2BodyDef();
		mBodyDef.set_type(box2d.b2_kinematicBody);
		mBodyDef.set_position(bi.body.GetPosition());

		// Store the body in case we want to control it later
		bi.motorBody = world.CreateBody(mBodyDef);
		box2d.destroy(mBodyDef);

		let md = new box2d.b2MotorJointDef();
		md.Initialize(bi.body, bi.motorBody);

		let mass = bi.body.GetMass();
		let inertia = bi.body.GetInertia();

		// Set force to raise it at the acceleration of gravity (with motorStrength == 1)
		md.set_maxForce(bi.motorStrength * mass * vec2.magnitude(bi.gravVec) * 2);

		// Set torque to spin it at 4 PI radians/sec (with motorStrength == 1)
		md.set_maxTorque(bi.motorStrength * inertia * 4 * Math.PI);

		bi.motorJoint = box2d.castObject(world.CreateJoint(md), box2d.b2MotorJoint);
		box2d.destroy(md);
	}
}

function createBox2dBody (self, args, bi, world) {
	// First figure out the geometry

	if (bi.paramOnly) return;

	const layer = bi.layer.privateLayer,
		  handle = layer.getHandleTreeRoot(),
		  matScene_Layer = args.getLayerMatrixRelativeToScene(layer),
		  matLayer_Handle = tasks.handle.getFrameRelativeToLayer(handle),
		  matScene_Handle = mat3.multiply(matScene_Layer, matLayer_Handle),
		  vecPuppet_Handle = handle.getMatrixAtRestRelativeToParent().getTranslation();

	// Save all handle transforms
	bi.handleTransforms = new Map();
	handle.breadthFirstEach(function (handleBody) {
		bi.handleTransforms.set(handleBody, tasks.handle.getFrame(handleBody));
	});

	let layerScale = [],
		layerRot = [],
		layerShear = [],
		layerPosition = [];

	mat3.decomposeAffine(matScene_Handle, layerPosition, layerScale, layerShear, layerRot);
	layerRot = (layerRot[2] < 0 ? -1 : 1) * Math.acos(layerRot[0]);

	// keep track of the scale so we can apply it when sending transform back to handles
	bi.scale = layerScale;
	layerPosition = self.convertPointToWorld(layerPosition);

	// points from container mesh are expressed relative to Source/Puppet coordinate frame...
	let meshPoints = [];
	if (bi.deformable) {
		meshPoints = layer.getWarperContainer().getOutlineWarpGeometry();
	} else {
		//meshPoints = layer.getWarperContainer().getMeshOutline();
		meshPoints = layer.getWarperContainer().getGeometryOutline();
	}

	// we need points to be relative to local/body frame which is ultimately in box2d world frame
	let polyPoints = [];

	// TODO: figure out why this is necessary, the two calls seem to return the points in different frames of reference
	if (bi.deformable) {
		polyPoints = meshPoints.map(function (cv/*, ind, arr*/) {
										// convert to box2d world
										return self.convertPointToWorld(cv);
									});
	} else {
		polyPoints = meshPoints.map(function (cv/*, ind, arr*/) {
										// convert from puppet to body frame and then to box2d world
										return self.convertPointToWorld(vec2.subtract(cv, vecPuppet_Handle));
									});
	}

	// Need to scale the input points, because box2d doesn't handle scaling
	polyPoints.forEach(function (p, ind, arr) {
		arr[ind] = [p[0] * layerScale[0], p[1] * layerScale[1]];
	});

	let tmpConvexPolys = [],
		circle = [],
		useCircle = false;

	if (bi.approxShape && polyPoints.length > 4) {
		// Compute convex hull, then use either enclosing circle or enclosing rectangle, whichever has minimum area
		let cHull = hull(polyPoints, Math.Infinity);

		// Remove the first point (it's repeated as the last point)
		cHull.splice(0, 1);

		circle = minEnclosingCircle(cHull);
		let minRect = minEnclosingBox(cHull);

		if (minRect.area < Math.PI * Math.pow(circle.radius, 2)) {
			tmpConvexPolys = [minRect.rect];
			//console.logToUser("Using rectangle");
		} else {
			useCircle = true;
			//console.logToUser("Using circle");
		}
	} else {
		// for deformables, store the previous position in the polyPoints array,
		// so velocities can be estimated later
		if (bi.deformable) {
			if (lodash.has(bi, "prevPoints")) {
				Z.utils.assert(bi.prevPoints.length === polyPoints.length, "prevPoints.length == polyPoints.length");

				for (let i = 0; i < polyPoints.length; ++i) {
					polyPoints[i].push(bi.prevPoints[i].slice(0, 2));
				}
			}
			else {
				for (let i = 0; i < polyPoints.length; ++i) {
					polyPoints[i].push(polyPoints[i].slice(0, 2));
				}
			}

			// Slice call to copy the array
			bi.prevPoints = polyPoints.slice(0);
		}

		// Split the shape into non-overlapping polygons
		let splitPolys = splitPolygonNonoverlapping(polyPoints);

		// Ensure correct winding
		splitPolys.forEach(function (p) {polyDecomp.makeCCW(p);});

		// Remove near-collinear points
		const angleThresh = 0.5 * Math.PI / 180.0; // radians
		splitPolys.forEach(function (p) {
			polyDecomp.removeCollinearPoints(p, angleThresh);
		});

		// Convex decomposition of input shape
		splitPolys.forEach(function (p) {
			let tmp = polyDecomp.quickDecomp(p); // fast algorithm
			//let tmp = polyDecomp.decomp(p); // slower, but optimal, algorithm

			tmp.forEach(function (t) {
				if (polygonArea(t) > 1e-3) {
					tmpConvexPolys.push(t);
				}
			});
		});

		if (!tmpConvexPolys || tmpConvexPolys.length === 0) {
			let cHull = hull(polyPoints, Math.Infinity);

			// Remove the first point (it's repeated as the last point)
			cHull.splice(0, 1);

			circle = minEnclosingCircle(cHull);
			useCircle = true;
		}
	}

	// Now create the body with the geometry

	if (bi.deformable) {
		// For deformable objects, create one box2d body for each convex section (with its own velocity)
		bi.childPieces = [];

		if (!useCircle) {
			tmpConvexPolys.forEach(function (poly) {
				if (poly.length === 0) return;

				let tmpBi = cloneBodyContainer(bi);

				createBox2dBodyHelper(self, args, tmpBi, world, {useCircle: useCircle, circle: circle, polygons:[poly], rotation: layerRot, position: layerPosition});

				bi.childPieces.push(tmpBi);
			});
		} else {
			let tmpBi = cloneBodyContainer(bi);

			createBox2dBodyHelper(self, args, tmpBi, world, {useCircle: useCircle, circle: circle, polygons:[], rotation: layerRot, position: layerPosition});

			bi.childPieces.push(tmpBi);
		}
	}
	else {
		// For rigid objects, create just one body, with multiple convex pieces
		createBox2dBodyHelper(self, args, bi, world, {useCircle: useCircle, circle: circle, polygons:tmpConvexPolys, rotation: layerRot, position: layerPosition});
	}
}

let jointType = {FREE: 0, HINGE: 1, WELD: 2};

function createHingeJoints(self, args, bi, world, mBodies) {
	if (bi.deformable || bi.paramOnly || !bi.body) return;

	// ... position
	const childLayer = bi.layer.privateLayer,
		parentLayer = getParentBodyLayer(childLayer);

	// Determine attachment type
	let layerMotionType = jointType.WELD;
	if (childLayer.motionMode.translation === false && childLayer.motionMode.linear === true) {
		layerMotionType = jointType.HINGE;
	} else if (childLayer.motionMode.translation === true && childLayer.motionMode.linear === true) {
		layerMotionType = jointType.FREE;
	}

	if (parentLayer) {
		const handleBody = bi.layer.privateLayer.getHandleTreeRoot(),
			  matLayer_Scene = args.getLayerMatrixRelativeToScene(bi.layer.privateLayer),
			  matScene_Joint = mat3.multiply(matLayer_Scene, tasks.handle.getFrameRelativeToLayer(handleBody)),
			  vecScene_Joint = matScene_Joint.getTranslation(),
			  worldPos = box2d.b2Vec2.from(self.convertPointToWorld(vecScene_Joint));

		let parentBi = mBodies.get(parentLayer.getSdkLayer()),
			parentBody = parentBi ? parentBi.body : false;

		if (!parentBody) {
			// Need to create a ghost body to control the joint
			let jointScale = [],
				jointRot = [],
				jointShear = [],
				jointPosition = [];

			mat3.decomposeAffine(matScene_Joint, jointPosition, jointScale, jointShear, jointRot);
			jointRot = (jointRot[2] < 0 ? -1 : 1) * Math.acos(jointRot[0]);

			let bodyDef = new box2d.b2BodyDef();
			bodyDef.set_position(worldPos);
			bodyDef.set_angle(jointRot);
			bodyDef.set_type(box2d.b2_staticBody);
			bi.hingeJointBody = world.CreateBody(bodyDef);
			parentBody = bi.hingeJointBody;
		}

		// Add constraints
		let jd = 0;

		if (layerMotionType === jointType.HINGE) {
			jd = new box2d.b2RevoluteJointDef();
		} else if (layerMotionType === jointType.WELD) {
			jd = new box2d.b2WeldJointDef();
		}

		if (jd) {
			jd.Initialize(bi.body, parentBody, worldPos);
			world.CreateJoint(jd);
			box2d.destroy(jd);
		} else if (bi.hingeJointBody) {
			world.DestroyBody(bi.hingeJointBody);
			bi.hingeJointBody = undefined;
		}

		box2d.destroy(worldPos);
	}
}

function createWalls (self, xs, ys, world) {
	// 0,0 is in the center of the screen, y increases downward
	let cornersPix = [vec2([-xs, ys]), vec2([-xs, -ys]), vec2([xs, ys]), vec2([xs, -ys])];
	let corners = cornersPix.map(function(x) { return self.convertPointToWorld(x);} );

	// create walls
	const fixtureDef = new box2d.b2FixtureDef();
	fixtureDef.set_friction(1); // Combined with geometric mean in contacts
	fixtureDef.set_restitution(0); // Combined with max in contacts
	// Walls are category 1 and collide with everything
	fixtureDef.get_filter().set_categoryBits(0x0001);
	fixtureDef.get_filter().set_maskBits(0xFFFF);
	let bodyDef = new box2d.b2BodyDef();
	let ground = world.CreateBody( bodyDef );
	box2d.destroy(bodyDef);

	let shape = new box2d.b2EdgeShape();
	let v1 = new box2d.b2Vec2(corners[0][0], corners[0][1]);
	let v2 = new box2d.b2Vec2(corners[1][0], corners[1][1]);
	shape.Set(v1, v2);
	fixtureDef.set_shape(shape);
	ground.CreateFixture(fixtureDef);
	box2d.destroy(shape);

	shape = new box2d.b2EdgeShape();
	v1.set_x(corners[0][0]); v1.set_y(corners[0][1]);
	v2.set_x(corners[2][0]); v2.set_y(corners[2][1]);
	shape.Set(v1, v2);
	fixtureDef.set_shape(shape);
	ground.CreateFixture(fixtureDef);
	box2d.destroy(shape);

	shape = new box2d.b2EdgeShape();
	v1.set_x(corners[1][0]); v1.set_y(corners[1][1]);
	v2.set_x(corners[3][0]); v2.set_y(corners[3][1]);
	shape.Set(v1, v2);
	fixtureDef.set_shape(shape);
	ground.CreateFixture(fixtureDef);
	box2d.destroy(shape);

	shape = new box2d.b2EdgeShape();
	v1.set_x(corners[2][0]); v1.set_y(corners[2][1]);
	v2.set_x(corners[3][0]); v2.set_y(corners[3][1]);
	shape.Set(v1, v2);
	fixtureDef.set_shape(shape);
	ground.CreateFixture(fixtureDef);
	box2d.destroy(shape);
	box2d.destroy(fixtureDef);

	box2d.destroy(v1);
	box2d.destroy(v2);
}

function setGravity (args) {
	// Apply gravity to each body
	let mBodies = getBodyMap(args.stageLayer);
	mBodies.forEach(function(bi) {
		if (bi.body && bi.dynamic && !bi.paramOnly) {
			let mass = bi.body.GetMass(),
				gravForce = new box2d.b2Vec2(bi.gravVec[0] * mass, bi.gravVec[1] * mass);

			bi.body.ApplyForceToCenter(gravForce, true);
			box2d.destroy(gravForce);
		}
	});
}

function validRigidBody (layer) {

	if (layer.privateLayer.getWarpWithParent()) {
		// Warp dependent layers are just storing params
		return true;
	}

	// Check if group has layers
	// Currently doing this by getting the outline and ensuring it has area
	// FIXME: There has to be a cleaner way to do this
	let meshPoints = layer.privateLayer.getWarperContainer().getGeometryOutline();

	let minp = 0,
	maxp = 0;

	if (meshPoints.length > 0) {
		minp = meshPoints[0][0];
		maxp = meshPoints[0][0];

		meshPoints.forEach(function (p) {
			if (p[0] < minp) minp = p[0];
			if (p[0] > maxp) maxp = p[0];
		});
	}

	if (minp === maxp) {
		return false;
	}

	return true;
}

// FIXME: Storing a global list of worlds here so that they can be deleted on a scene refresh. Each scene level behavior will push its world onto this list.
// This will need to be fixed if we ever allow multiple scenes to run simultaneously.
let worlds = [];

function setupBodyParams (layer, mData, params, dangleLayers) {

	mData.dynamic = mData.hasOwnProperty("dynamic") ? mData.dynamic : false;
	mData.collide = mData.collide || false;

	// Layers with dangle handles can't be dynamic
	if (dangleLayers && dangleLayers.has(layer)) {
		mData.dynamic = false;
	}

	// Dependent puppets, or the root puppet, are only param holders
	// Also, if it is not dynamic or collide (can happen when a dynamic only layer also has dangle handles), then make it param only.
	let paramOnly = layer.privateLayer.getWarpWithParent() || !layer.privateLayer.parentPuppet || (!mData.dynamic && !mData.collide);

	let bodyDef = {
		body : false,
		layer : layer,
		trackItem : paramOnly ? false : layer.privateLayer.getTrackItem(),
		paramOnly : paramOnly,
		density : params.density,
		dynamic : mData.dynamic,
		collidable : mData.collide,
		autoAddChildren : false,
		// FIXME: currently looking at the privateWarper object to see how many anchors (handles) there are.
		// Deformable bodies need more than one. Should this be a cpp callback, or just store it in the regular warperContainer object instead of the private one?
		deformable : !mData.dynamic && mData.collide,
		motorStrength : params.motorStrength,
		motorBody : undefined,
		//source : false,
		approxShape : params.approxShape,
		friction : params.friction,
		restitution : params.restitution,
		wallCollide : params.wallCollide,
		beingAltered : new Map(),
		action : params.action,
		controlJoints : new Map(),
		controlJointBody : undefined,
		beingDragged : new Map(),
		behaviorLayer : params.behaviorLayer,
		gravVec : params.gravVec
	};

	return bodyDef;
}

function updateBodyListAndParams (self, args) {
	let tBox2d = args.stageLayer.getSharedSceneData(kPrefix);

	var time = args.timeStack.getRootTimeSpan().t;
	args.forEachTrackItemPuppet(time, function (rootSdkLayer) {
		rootSdkLayer.forEachLayerBreadthFirst(function (sdkLayer) {
			let layer = sdkLayer.privateLayer;

			// skip over layers without display container
			let displayContainer = layer.getDisplayContainer();
			if (! displayContainer) return;

			let warperLayer = layer.getWarperLayer();
			if (! warperLayer) return;

			// skip over Layers without warper container
			let warperContainer = warperLayer.getWarperContainer();
			if (! warperContainer) return;

			// skip if no warping
			let warperContainerCanWarp = warperContainer.canWarp();
			if (! warperContainerCanWarp) return;

			// skip layers that aren't in our scene
			if (warperLayer.getScene() !== args.stageLayer.privateLayer.getScene()) return;

			// Check if it does not have parameters
			let tags = layer.getLayerTags(),
				tagDynamic = !lodash.isUndefined(tags["Adobe.Physics.Dynamic"]),
				tagCollide = !lodash.isUndefined(tags["Adobe.Physics.Collide"]);

			// Find closest parent and use its params
			if ((tagDynamic || tagCollide)) {
				let mData = {dynamic: tagDynamic, collide: tagCollide};

				let curLayer = sdkLayer,
					firstIndepLayer = layer.getWarpWithParent() ? false : curLayer,
					params = tBox2d.bodies.get(curLayer);

				while ((!params || !params.attachedToBehavior) && curLayer) {
					curLayer = curLayer.getParentLayer();
					params = tBox2d.bodies.get(curLayer);

					if (!firstIndepLayer && !curLayer.privateLayer.getWarpWithParent()) {
						firstIndepLayer = curLayer;
					}
				}

				if (params && !params.attachedToBehavior) {
					params = undefined;
				}

				// Keep going if we didn't find an independent layer
				while (!firstIndepLayer && curLayer) {
					curLayer = curLayer.getParentLayer();

					if (!curLayer.privateLayer.getWarpWithParent()) {
						firstIndepLayer = curLayer;
					}
				}

				if (!firstIndepLayer.privateLayer.parentPuppet) {
					// There are no independent layers in this layer's parents
					return;
				}

				if (self.world && params && firstIndepLayer) {
					// Update params here
					let tBox2d = args.stageLayer.getSharedSceneData(kPrefix);
					let bodyDef = tBox2d.bodies.get(firstIndepLayer);

					bodyDef.gravVec = params.gravVec;
					bodyDef.friction = params.friction;
					bodyDef.restitution = params.restitution;
					bodyDef.density = params.density;
					bodyDef.motorStrength = params.motorStrength;
					bodyDef.wallCollide = params.wallCollide;
					bodyDef.action = params.action;

					tBox2d.bodies.set(firstIndepLayer, bodyDef);
					args.stageLayer.setSharedSceneData(kPrefix, tBox2d);

					if (!bodyDef.body && !bodyDef.childPieces) {
						return;
					}

					// Update friction & restitution in the body
					let updateFunc = function (b) {
						let fix = b.GetFixtureList();
						while (box2d.getPointer(fix)) {
							fix.SetDensity(bodyDef.density);
							fix.SetFriction(bodyDef.friction);
							fix.SetRestitution(bodyDef.restitution);

							let fd = fix.GetFilterData();

							if (bodyDef.wallCollide) {
								fd.set_maskBits(0x0003);
							} else {
								fd.set_maskBits(0x0002);
							}

							fix.SetFilterData(fd);

							fix = fix.GetNext();
						}

						b.ResetMassData();
					};

					if (bodyDef.body) {
						updateFunc(bodyDef.body);
					} else if (bodyDef.childPieces) {
						// Deformable, update for child pieces
						bodyDef.childPieces.forEach(function (cp) {
							updateFunc(cp.body);
						});
					}

					if (bodyDef.motorJoint) {
						let maxForce = bodyDef.motorStrength * bodyDef.body.GetMass() * vec2.magnitude(bodyDef.gravVec) * 2,
							maxTorque = bodyDef.motorStrength * bodyDef.body.GetInertia() * 4 * Math.PI;

						bodyDef.motorJoint.SetMaxForce(maxForce);
						bodyDef.motorJoint.SetMaxTorque(maxTorque);
					}
				} else {
					let addToTreeOrder = function (layerToAdd) {
						let found = false;
						self.treeOrder.forEach(function (l) {
							if (l === layerToAdd) found = true;
						});

						if (!found) {
							self.treeOrder.push(layerToAdd);
						}
					};

					if (!tBox2d.bodies.has(firstIndepLayer) && params && firstIndepLayer && validRigidBody(firstIndepLayer)) {
						let body = setupBodyParams(firstIndepLayer, mData, params, tBox2d.dangleLayers);

						addToBodyList(args.stageLayer, body);

						addToTreeOrder(firstIndepLayer);
					} else if (tBox2d.bodies.has(firstIndepLayer) && !tBox2d.bodies.get(firstIndepLayer).paramOnly) {
						addToTreeOrder(firstIndepLayer);
					}
				}
			}
		});
	},
	args.stageLayer.privateLayer.getScene());
}

function init (self, args) {
	self.treeOrder = [];
	updateBodyListAndParams(self, args);

	// TODO: try moving to onCreateStageBehavior()
	const size = getSceneScale(args), invSize = 1.0 / size;

	// TODO: Could move into class instead of closures
	const matWorld_Frame = mat3.from([invSize, 0, 0, 0, -invSize, 0, 0, 0, 1]),
		matFrame_World = mat3.from([size, 0, 0, 0, -size, 0, 0, 0, 1]);
	self.convertLengthToWorld = function (size) { return size.map(function (s) { return invSize * s; }); };
	self.convertPointToWorld = vec2.transformLinear.bind(undefined, matWorld_Frame);
	self.convertPointFromWorld = vec2.transformLinear.bind(undefined, matFrame_World);
	self.convertFrameFromWorld = function (matWorld) { return mat3.multiply(matFrame_World, mat3.multiply(matWorld, matWorld_Frame)); };

	let tempGravity = new box2d.b2Vec2(0, 0);
	let world = new box2d.b2World(tempGravity);
	world.SetAutoClearForces(false);
	worlds.push(world);
	box2d.destroy(tempGravity);

	world.SetAllowSleeping(false);

	// create walls if requested
	if (args.getParam("walls")) {
		// get scene bounds
		let xs = args.scene_.getPixelWidth() / 2,
			ys = args.scene_.getPixelHeight() / 2;

		createWalls(self, xs, ys, world);
	}

	// import bodies for each puppet with Box2dBody behavior
	const sceneLayer = args.stageLayer;
	let mBodies = getBodyMap(sceneLayer);

	if (mBodies) {
		// Create all the bodies in box2d
		mBodies.forEach(function (bi) {
			if (bi.action === kCreate) {
				createBox2dBody(self, args, bi, world);
				bi.action = kNone;
			}
		});

		updateBodyMap(sceneLayer, mBodies);

		// Final loop to create constraints
		mBodies.forEach(function (bi) {
			createHingeJoints(self, args, bi, world, mBodies);
		});

		updateBodyMap(sceneLayer, mBodies);
	}
	self.world = world;

	self.lastBirth = -1;
	self.lastT = -1;
	self.lastDown = -1;
	self.timeResidual = 0.0;
}

function updateDeltaT (self, args) {
	// Time book-keeping
	const currT = Math.max(0.0, args.t) + args.globalRehearseTime;
	// FIXME: why is args.t < 0 on init now?
	//Z.utils.assert(args.t >= 0, "args.t < 0 at value: " + args.t);

	if (self.lastT < 0.0 ) {
		self.lastT = currT;
	}

	self.deltaT = currT - self.lastT;
	self.lastT = currT;

	// Make sure we always add at least the fps time step, because globalRehearseTime and args.t
	// only work correctly when running live, not during recording playback
	// TODO: limit the time residual so box2d doesn't slow anything down?
	//self.timeResidual += Math.max(self.deltaT, 1.0 / args.scene_.getFrameRate());

	// Step at fixed rate for now
	self.timeResidual += 1.0 / args.scene_.getFrameRate();
}

function updateControlJoints (args, self, world, mBodies) {
	mBodies.forEach(function (bi) {
		if (bi.deformable || bi.paramOnly || !bi.body) return;

		const rootHandleBody = bi.layer.privateLayer.getHandleTreeRoot(),
			  matScene_Layer = args.getLayerMatrixRelativeToScene(bi.layer.privateLayer);

		rootHandleBody.breadthFirstEach(function (handleBody) {
			if (bi.beingDragged.get(handleBody)) {
				bi.beingDragged.set(handleBody, false);

				let handleFrame = tasks.handle.getFrameRelativeToLayer(handleBody),
					target = mat3.multiply(matScene_Layer, handleFrame).getTranslation();

				if (bi.beingAltered.get(handleBody)) {
					// Update control target position
					let tmpPos = box2d.b2Vec2.from(self.convertPointToWorld(target));
					bi.controlJoints.get(handleBody).SetTarget(tmpPos);
					box2d.destroy(tmpPos);
				} else {
					// Create a control joint
					let mj = new box2d.b2MouseJointDef();
					mj.set_bodyA(bi.controlJointBody);
					mj.set_bodyB(bi.body);
					let tmpPos = box2d.b2Vec2.from(self.convertPointToWorld(target));
					mj.set_target(tmpPos);
					box2d.destroy(tmpPos);
					mj.set_maxForce(1000 * bi.body.GetMass());
					//mj.set_frequencyHz(10);
					//mj.set_dampingRatio(1);

					bi.controlJoints.set(handleBody, box2d.castObject(world.CreateJoint(mj), box2d.b2MouseJoint));
					bi.beingAltered.set(handleBody, true);
					box2d.destroy(mj);
				}
			} else if (bi.beingAltered.get(handleBody)) {
				// Destroy control joint if it exists
				world.DestroyJoint(bi.controlJoints.get(handleBody));
				bi.controlJoints.set(handleBody, undefined);
				bi.beingAltered.set(handleBody, false);
			}
		});
	});
}

// For joints that are not attached to a parent box2d body,
// we need to manually update the ghost body
function updateAttachJoints (args, self, world, mBodies) {
	mBodies.forEach(function (bi) {
		if (bi.deformable || bi.paramOnly || !bi.body) return;

		if (bi.hingeJointBody) {
			let handleBody = bi.layer.privateLayer.getHandleTreeRoot(),
				matScene_Layer = args.getLayerMatrixRelativeToScene(bi.layer.privateLayer),
				matLayer_Handle = tasks.handle.getFrameRelativeToLayer(handleBody),
				matScene_Handle = mat3.multiply(matScene_Layer, matLayer_Handle);

			let jointScale = [],
				jointRot = [],
				jointShear = [],
				jointPosition = [];

			mat3.decomposeAffine(matScene_Handle, jointPosition, jointScale, jointShear, jointRot);
			jointRot = (jointRot[2] < 0 ? -1 : 1) * Math.acos(jointRot[0]);

			let worldPos = self.convertPointToWorld(jointPosition),
				worldVec = box2d.b2Vec2.from(worldPos);

			bi.hingeJointBody.SetTransform(worldVec, jointRot);

			//console.logToUser(`${worldPos}, ${jointRot}`);

			box2d.destroy(worldVec);
		}
	});
}

function destroyBody (self, bi) {
	if (bi.childPieces) {
		bi.childPieces.forEach(function (tmpBi) {
			self.world.DestroyBody(tmpBi.body);
			self.world.DestroyBody(tmpBi.controlJointBody);
			if (tmpBi.motorBody) {
				self.world.DestroyBody(tmpBi.motorBody);
			}

			if (tmpBi.hingeJointBody) {
				self.world.DestroyBody(tmpBi.hingeJointBody);
			}

			tmpBi.body = false;
			tmpBi.controlJointBody = false;
			tmpBi.motorBody = false;
			tmpBi.hingeJointBody = undefined;
			tmpBi.beingAltered = new Map();
			tmpBi.prevTarget = undefined;
			tmpBi.controlJoints = new Map();
			tmpBi.motorJoint = undefined;
		});
	}

	bi.childPieces = undefined;

	if (bi.body) {
		self.world.DestroyBody(bi.body);
		self.world.DestroyBody(bi.controlJointBody);
		if (bi.motorBody) {
			self.world.DestroyBody(bi.motorBody);
		}

		if (bi.hingeJointBody) {
			self.world.DestroyBody(bi.hingeJointBody);
		}

		bi.body = false;
		bi.controlJointBody = false;
		bi.motorBody = false;
		bi.hingeJointBody = undefined;
		bi.beingAltered = new Map();
		bi.prevTarget = undefined;
		bi.controlJoints = new Map();
		bi.beingDragged = new Map();
		bi.motorJoint = undefined;
	}
}

function recreateDeformables (self, args) {
	const sceneLayer = args.stageLayer;

	let mBodies = getBodyMap(sceneLayer);

	if (!mBodies) return;

	mBodies.forEach(function (bi) {
		if (bi.deformable && !bi.paramOnly && bi.childPieces) {
			// Destroy this body and create it again to get deformed geometry

			destroyBody(self, bi);

			createBox2dBody(self, args, bi, self.world);
		}
	});
}

function handleTriggers (self, args) {
	let mBodies = getBodyMap(args.stageLayer);

	let newBodies = new Map();

	mBodies.forEach(function (bi) {
		if (bi.action === kCreate) {
			createBox2dBody(self, args, bi, self.world);
			newBodies.set(bi.layer, true);
		} else if (bi.action === kDestroy) {
			destroyBody(self, bi);
		}

		bi.action = kNone;
	});

	mBodies.forEach(function (bi) {
		if (newBodies.has(bi.layer)) {
			createHingeJoints(self, args, bi, self.world, mBodies);
		}
	});

	updateBodyMap(args.stageLayer, mBodies);
}

function step (self, args) {
	updateBodyListAndParams(self, args);

	const world = self.world,
		timeStep = 1.0 / args.getParam("fps"),
		velocityIterations = 10,
		positionIterations = 10;

	updateDeltaT(self, args);

	// Create/destroy bodies if triggered
	handleTriggers(self, args);

	// Recreate deformable bodies
	recreateDeformables(self, args);

	setGravity(args);

	while (self.timeResidual >= timeStep) {
		world.Step(timeStep, velocityIterations, positionIterations);
		self.timeResidual -= timeStep;
	}

	world.ClearForces();

	let mBodies = getBodyMap(args.stageLayer);
	if (mBodies) {
		//mBodies.forEach(function (bi) {
		//    let layer = bi.trackItem.getSdkLayer().privateLayer;
		//    console.logToUser(`pre: ${layer.tomNow}`);
		//});

		updateControlJoints(args, self, world, mBodies);

		updateAttachJoints(args, self, world, mBodies);

		var tTrackItemLayers = {}; // unique set of track items indexed by stage id

		// Set layer transforms in tree order (parents before children)
		self.treeOrder.forEach(function (layerInOrder) {
			let bi = mBodies.get(layerInOrder);

			if (bi.deformable || bi.paramOnly || !bi.body) return;

			const body = bi.body,
				  layer = bi.layer.privateLayer,
				  handle = layer.getHandleTreeRoot(),
				  position = body.GetPosition().asArray(),
				  angle = body.GetAngle(),
				  trackItem = bi.trackItem;

			let matWorld_Body = mat3.identity().translate(position).rotate(angle).scale(bi.scale),
				matScene_Body = self.convertFrameFromWorld(matWorld_Body);

			//console.logToUser(`setting layer '${layer.getStageId()}' with path '${tasksInternal.getKeyPath(layer).key}'`);

			const layerTrack = trackItem.getSdkLayer().privateLayer,
			matTrack_Scene = mat3.invert(args.getLayerMatrixRelativeToScene(layerTrack)),
			matTrack_Body = mat3.multiply(matTrack_Scene, matScene_Body);

			// console.logToUser(`matScene_Body: ${matScene_Body}`);
			// console.logToUser(`matTrack_Body: ${matTrack_Body}`);

			// Rember layers we need to commit
			tTrackItemLayers[trackItem.getStageId()] = layerTrack;

			const matLayer_Body = tasksInternal.convertTrackLayerFrame(layer, layerTrack, matTrack_Body);
			// console.logToUser(`matLayer_Body: ${matLayer_Body}`);

			const matHandle_Body = tasks.handle.convertLayerFrame(handle, matLayer_Body);
			//console.logToUser(`matHandle_Body: ${matHandle_Body}`);

		   // For dynamic layers, we need to set all the child handles to their original transform (not allowed to change shape)
			handle.breadthFirstEach(function (handleBody) {
				if (handleBody === handle) return;

				tasks.handle.setFrame(handleBody, bi.handleTransforms.get(handleBody), tasks.dofs.type.kAffine);
			});

			tasks.handle.setFrame(handle, matHandle_Body, tasks.dofs.type.kAffine);
		});

		lodash.forOwn(tTrackItemLayers, (layer) => {layer.commitMatrixDofs();});

		mBodies.forEach(function (bi) {
			if (bi.deformable || bi.paramOnly) return;

			let layer = bi.layer.privateLayer,
				rootHandle = layer.getHandleTreeRoot();

			bi.prevMDof = new Map();

			rootHandle.breadthFirstEach(function (handle) {
				bi.prevMDof.set(handle, tasks.handle.internal.getMatrixDof(handle));
			});
		});

		updateBodyMap(args.stageLayer, mBodies);
	}
}

return {
	about: "$$$/private/animal/Behavior/Box2d/World/About=Box2d World.",
	description: "$$$/private/animal/Behavior/Box2d/World/Desc=Rigid-Body World",
	uiName: "$$$/private/animal/Behavior/Box2d/World/UIName=Box2dWorld",
	defaultArmedForRecordOn: true,
	applyToSceneOnly: true,

	defineParams: function () {
		// free function, called once ever; returns parameter definition (hierarchical) array
		return [
			{
				id: "walls", type: "checkbox", uiName: "$$$/private/animal/Behavior/Box2d/World/Parameter/walls=Add Walls", dephault:true
			},
			{
				id: "fps", type: "slider", uiName: "$$$/private/animal/Behavior/Box2d/World/Parameter/fps=Simulation FPS", min: 60, precision: 1, max: 1000, dephault: 60
			},
		];
	},

	onCreateBackStageBehavior: function (/*self*/) {
		return {
			order : 1.0,
			importance : 1.0
		};
	},

	terminateStage: function () {
		worlds.forEach(function(world) {
			box2d.destroy(world);
		});

		worlds = [];
	},

	onCreateStageBehavior: function (self, args) {
		args.addSceneTerminationFunction(this.terminateStage);
	},

	onResetRehearsalData : function (/*self,args*/) {
	},

	onAnimate: function (self, args) {
		const sceneLayer = args.stageLayer;
		let mBodies = getBodyMap(sceneLayer);

		if (mBodies && mBodies.size > 0) {
			if (self.world) {
				step(self, args);
			} else {
				init(self, args);
			}
		}
	}

}; // end of object being returned

});
